//: ## ![3DaysOfSwift.com Logo](3DaysIcon46.png) Frameworks
//:
//: ## Task:
//:
//: Many different types exist outside of Swift.
//:
//: In fact, Apple created "framework" files to contain a collection of code to generate more types and extend features of the language to better suit and interact with thier own devices.
//:
//: We can import frameworks by using the `import` keyword at the beginning of your code.
//:
/*:
 
 */
//: Resolve the compilation error below by uncommenting out the line of code that imports Apples `Foundation` framework.
//:
//: -------------------
//:
//: [◀ Previous Page](@previous) | [Next Page  ▶](@next)
//:
//: -------------------
//:


// import Foundation
//let anAccurateDecimal: Decimal = 0.4 + 0.2
//let todaysDate: Date = Date()
//let coordinate: CGPoint = CGPoint(x: 100.0,y: 100.0)
//let gregorianCalendar: Calendar = Calendar(identifier: .gregorian)


//:
//: -------------------
//:
//: ## Date
//:
//: The `Foundation` framework contains the `Date` type.
//:
//: This is a very powerful type that helps us navigate our way through our complex calendar systems.
//:
//: -------------------
//:



import Foundation
struct LeapYearTracker {
    var numberOfYearsToCheckInFuture: Int = 8 // check 8 years into the future
    func nextLeapYear() -> Date? {
        let dateNow = Date()
        let calendar = Calendar.current
        let currentYear = calendar.component(.year, from: dateNow)
        
        for year in (currentYear + 1)...(currentYear + numberOfYearsToCheckInFuture) {
            let isLeapYear = isLeapYearFormulae(year)
            if isLeapYear {
                return leapYearDate(from: year, calendar: calendar)
            }
        }
        return nil // Couldn't find one
    }
    
    func isLeapYearFormulae(_ year: Int) -> Bool {
        return (year % 4 == 0) && ((year % 100 != 0) || (year % 400 == 0))
    }
    
    func leapYearDate(from year: Int, calendar: Calendar) -> Date? {
        var components = DateComponents()
        components.year = year
        components.month = 2
        components.day = 29
        return calendar.date(from: components)
    }
}

let leapYearTracker = LeapYearTracker()
if let leapDate = leapYearTracker.nextLeapYear() {
    let formatter = DateFormatter()
    formatter.dateStyle = .full
    print("The next leap day is on \(formatter.string(from: leapDate))")
} else {
    print("Could not find the next leap year.")
}


//:
//: -------------------
//:
//: ## Task:
//:
//: Using the code above as a guide, create an extension on the `Date` type to check if a date falls within a leap year.
//:
//: -------------------
//:
//: [◀ Previous Page](@previous) | [Next Page  ▶](@next)
//:
//: -------------------
//:


// write code here


//:
//: -------------------
//:
//: ## Task:
//:
//: The code above is a little unusual because it contains two seperate methods for checking if a date is a leap year.
//:
//: Considering that the instance of the `DateComponents` type will only return a new instance of a `Date` if the special leap year date of 29th Feb exists, perhaps we should use this as our only check?
//:
//: Some points to consider:
//:
//: * if we iterate over integers then our code will be very fast.
//:
//: * if we iterate over potentially instantiating `Date` objects then it will require more calculation and require more CPU time as a result.
//:
//: * if we apply two separate methods of calculating if a leap year is a leap year then we may get it wrong and miss the opportunity to check the `DateComponents` if it actually exists.
//:
//:
//: -------------------
//:
//: ## Task:
//:
//: Within a new `Date` extension, create a new function named `nextLeapYear` that'll generate the next leap year from itself.
//:
//: -------------------
//:
//: [◀ Previous Page](@previous) | [Next Page  ▶](@next)
//:
//: -------------------
//:


// write extension here

let fiveYearsFromNowInSeconds: TimeInterval = 60 * 60 * 24 * 31 * 12 * 5
let date: Date = Date(timeIntervalSinceNow: fiveYearsFromNowInSeconds)
let nextLeapYear: Date = date.nextLeapYear() // name your function nextLeapYear


//:
//: -------------------
//:
/*:
  * Callout(Tip 💡):
 Did you know that `TimeInterval` is actually a `Double`?
 
    Apple created a "type alias" to provide more clarity and readability to our code. It suggests the double value will represent time in seconds.
 */
//:
//: We don't have to calculate the year based on seconds.
//:
//: Read the code below. We use the Calendar type to add one year to todays date.
//:
//: -------------------
//:


let dateNow: Date = Date()
if let dateInOneYear: Date = Calendar.current.date(byAdding: .year, value: 1, to: dateNow) {
    let nextLeapYear: Date = dateInOneYear.nextLeapYear()
}


//:
//: -------------------
//:
/*:
  * Callout(Tip 💡):
    Did you notice one of our calculations is technically incorrect for calulculating the next year?
 
    We didn't take into consideration how many seconds in a year fluctuates.
 
    Using the `Calendar` type to generate the precise date would be a more appropriate and stable solution.
 */
//:
//: -------------------
//:
//: ## Task:
//:
//: Use the new `Calendar` extension below to print the date one year from now to the console window.
//:
//: -------------------
//:
//: [◀ Previous Page](@previous) | [Next Page  ▶](@next)
//:
//: -------------------
//:


extension Calendar {
    var oneYearFromNow: Date? {
        Calendar.current.date(byAdding: .year, value: 1, to: Date())
    }
}

// write code here

